"use client"

import { FadeIn } from "@/components/animations/fade-in";
import { Button } from "@/components/ui/button";
import { DropdownMenuContent, DropdownMenuGroup, DropdownMenuItem, DropdownMenuTrigger } from "@/components/ui/dropdown-menu";
import { Tooltip, TooltipContent, TooltipProvider, TooltipTrigger } from "@/components/ui/tooltip";
import { useApplicationWorkflow } from "@/hooks/use-application-workflow";
import { focusRing } from "@/lib/styles";
import { cn } from "@/lib/utils";
import { AppValidationError } from "@/zod/feeds.schema";
import { DropdownMenu } from "@radix-ui/react-dropdown-menu";
import { Check, ChevronDown, ExternalLink, Link, Loader, X, } from "lucide-react";
import { useState } from "react";
import { useStatusLoadingContent, UseStatusLoadingContentProps } from "../hook/use-status-loading-content";
import { useStatusLoadingMessages } from "../hook/use-status-loading-messages";
import { StatusBannerContentWrapper, StatusBannerIconWrapper, StatusBannerWrapper } from "./status.shared";

const INTERNAL__loadingContent = {
  loading: (m) => <INTERNAL__Loading agencies={m?.RUNNING.map((l) => l.gtfsId)} />,
  error: (m, e) => <INTERNAL__Error error={e} agencies={m?.FAILED.map((f) => f.gtfsId)} />,
} satisfies UseStatusLoadingContentProps

export function StatusRevalidateBanner({ delay }: { delay?: number }) {
  const {
    isLoading,
    data: showing
  } = useStatusLoadingContent(INTERNAL__loadingContent)

  if (isLoading || !showing) {
    return
  }

  return (
    <FadeIn delay={delay}>
      <StatusBannerWrapper>
        {showing}
      </StatusBannerWrapper>
    </FadeIn>
  )
}

function INTERNAL__Loading({
  agencies
}: {
  agencies: string[] | undefined,
}) {
  const message = useStatusLoadingMessages()
  return (
    <>
      <StatusBannerIconWrapper className="animate-pulse">
        <Loader className="opacity-60 animate-spin size-4 stroke-2" />
      </StatusBannerIconWrapper>

      <StatusBannerContentWrapper>
        <p className="text-sm font-medium">A build is currently in progress for{" "}
          {!!agencies && agencies.length >= 1 ? (
            <b>{agencies.join(', ')}</b>
          ) : (
            <>one or more agencies</>
          )}{" "}
          .</p>
        <p className="text-xs text-muted-foreground">{message}</p>
      </StatusBannerContentWrapper>

      <StatusBannerContentWrapper className="animate-pulse">
        <p className="text-sm font-medium"></p>
      </StatusBannerContentWrapper>
    </>
  )
}

// function INTERNAL__Success() {
//   const [date] = useState(new Date(Date.now()))
//   return (
//     <>
//       <StatusBannerIconWrapper>
//         <Check className="opacity-60 size-4 stroke-2 text-green-500 dark:text-green-600" />
//       </StatusBannerIconWrapper>
//
//       <StatusBannerContentWrapper>
//         <p className="text-sm font-medium">Open trip planner server was successfully built!</p>
//         <p className="text-xs text-muted-foreground">Successfully built at {date.toString()}</p>
//       </StatusBannerContentWrapper>
//     </>
//   )
// }
//
function INTERNAL__Error({
  agencies,
}: {
  error: AppValidationError,
  agencies: string[] | undefined,
}) {
  const [opened, setOpened] = useState<boolean>(false);
  const [openedAgency, setOpenedAgency] = useState<boolean>(false);
  const { setWorkflow } = useApplicationWorkflow()

  const handleOpenAgencyPage = (agency: string) => {
    if (!agency) {
      return;
    }

    setWorkflow({
      __tag: "STATUS_AGENCY_FEED",
      id: agency,
    })
    setOpenedAgency(true);
    setTimeout(() => setOpenedAgency(false), 750);
  };

  const handleOpen = () => {
    window.open("https://us-east-1.console.aws.amazon.com/cloudwatch/home?region=us-east-1#home:", "_blank");
    setOpened(true);
    setTimeout(() => setOpened(false), 1500);
  };

  return (
    <>
      <StatusBannerIconWrapper>
        <X className="opacity-60 size-4 stroke-2 text-destructive" />
      </StatusBannerIconWrapper>

      <StatusBannerContentWrapper>
        <p className="text-sm font-medium">Error occurred during build.</p>
        <p className="text-sm text-muted-foreground line-clamp-2">
          One or more feeds have failed during build.{" "}
          {!!agencies && agencies.length >= 1 && <>View the error logs for <b>{agencies.join(', ')}</b> by clicking the link.{" "}</>}
          Futher details can be viewed in the Amazon CloudWatch Console.{" "}
        </p>
      </StatusBannerContentWrapper>

      <div className="flex flex-col md:flex-row">
        {agencies && agencies.length === 1 && (
          <TooltipProvider delayDuration={0}>
            <Tooltip>
              <TooltipTrigger asChild>
                <Button
                  variant={"outline"}
                  onClick={() => handleOpenAgencyPage(agencies[0])}
                  className={cn(
                    focusRing,
                    "inline-flex items-center justify-center whitespace-nowrap text-sm font-medium disabled:pointer-events-none disabled:opacity-50 border border-transparent !outline-none focus-visible:!z-[100] transition-shadow",
                    "relative col-span-1 rounded-none",
                    "h-1/2 aspect-video border-s border-s-border border-b border-b-border rounded-se-xl focus-visible:!border-s focus-visible:!border-e",
                    "md:h-full md:max-w-32 md:aspect-square md:rounded-se-none md:border-b-transparent",
                    "group"
                  )}
                  aria-label={`Open agency ${agencies[0]} status page`}
                  disabled={openedAgency}
                >
                  <div className={"sr-only"}>
                    Open agency status page
                  </div>

                  <div
                    className={cn(
                      "absolute transition-all",
                    )}
                  >
                    <Link className="group-focus-visible:text-foreground text-muted-foreground size-3" aria-hidden="true" />
                  </div>
                </Button>
              </TooltipTrigger>
              <TooltipContent className="border border-input bg-popover px-2 py-1 text-xs text-muted-foreground">
                Open agency status page
              </TooltipContent>
            </Tooltip>
          </TooltipProvider>
        )}

        {agencies && agencies.length > 1 && (
          <TooltipProvider delayDuration={0}>
            <Tooltip>
              <DropdownMenu>
                <TooltipTrigger
                  className="h-1/2 md:h-full"
                >
                  <DropdownMenuTrigger
                    className={cn(
                      focusRing,
                      "border border-input bg-background hover:bg-accent hover:text-accent-foreground",
                      "inline-flex items-center justify-center whitespace-nowrap text-sm font-medium disabled:pointer-events-none disabled:opacity-50 border border-transparent !outline-none focus-visible:!z-[100] transition-shadow",
                      "relative col-span-1 rounded-none",
                      "h-full aspect-video border-s border-s-border border-b border-b-border rounded-se-xl focus-visible:!border-s focus-visible:!border-e",
                      "md:h-full md:max-w-32 md:aspect-square md:rounded-se-none md:border-b-transparent",
                      "group"
                    )}
                    aria-label={`Open agency ${agencies[0]} status page`}
                    disabled={openedAgency}
                  >
                    <div className={"sr-only"}>
                      Open agency status page
                    </div>

                    <div
                      className={cn(
                        "absolute transition-all",
                      )}
                    >
                      <Link className="group-focus-visible:text-foreground text-muted-foreground size-3" aria-hidden="true" />
                    </div>

                    <div
                      className={cn(
                        "absolute bottom-3 transition-all hidden md:block",
                      )}
                    >
                      <ChevronDown className="group-focus-visible:text-foreground text-muted-foreground size-3" aria-hidden="true" />
                    </div>
                  </DropdownMenuTrigger>
                </TooltipTrigger>
                <DropdownMenuContent>
                  <DropdownMenuGroup>
                    {agencies.map((a) => (
                      <DropdownMenuItem key={a} onClick={() => handleOpenAgencyPage(a)}>{a}</DropdownMenuItem>
                    ))}
                  </DropdownMenuGroup>
                </DropdownMenuContent>
              </DropdownMenu>
              <TooltipContent className="border border-input bg-popover px-2 py-1 text-xs text-muted-foreground">
                Open agency status page
              </TooltipContent>
            </Tooltip>
          </TooltipProvider>
        )}

        <TooltipProvider delayDuration={0}>
          <Tooltip>
            <TooltipTrigger asChild>
              <Button
                variant={"outline"}
                onClick={handleOpen}
                className={cn(
                  focusRing,
                  "inline-flex items-center justify-center whitespace-nowrap text-sm font-medium disabled:pointer-events-none disabled:opacity-50 border border-transparent !outline-none focus-visible:!z-[100] transition-shadow",
                  "relative col-span-1 bg-transparent rounded-none",
                  "h-1/2 aspect-video border-s border-s-border border-t border-t-border rounded-ee-xl focus-visible:!border-s focus-visible:!border-e",
                  "md:h-full md:max-w-32 md:aspect-square md:rounded-ee-none md:rounded-e-xl md:border-s md:border-s-border md:border-t-transparent",
                  "group focus-visible:!text-primary focus-visible:!border-primary ring-primary/20 "
                )}
                aria-label={opened ? "Copied" : "Copy to clipboard"}
                disabled={opened}
              >
                <div
                  className={cn(
                    "absolute transition-all",
                    opened ? "scale-100 opacity-100" : "scale-0 opacity-0",
                  )}
                >
                  <Check
                    className="stroke-emerald-500 size-3"
                    aria-hidden="true"
                  />
                </div>

                <div className={"sr-only"}>
                  Open in online CloudWatch Console
                </div>

                <div
                  className={cn(
                    "absolute transition-all",
                    !opened ? "scale-100 opacity-100" : "scale-0 opacity-0",
                  )}
                >
                  <ExternalLink className="group-focus-visible:text-foreground text-muted-foreground size-3" aria-hidden="true" />
                </div>
              </Button>
            </TooltipTrigger>
            <TooltipContent className="border border-input bg-popover px-2 py-1 text-xs text-muted-foreground">
              Open in online CloudWatch Console
            </TooltipContent>
          </Tooltip>
        </TooltipProvider>
      </div >
    </>
  )
}
